<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Handle cart actions
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    if(isset($_POST['remove_item'])) {
        $product_id = (int)$_POST['product_id'];
        // Remove from cart session
        if(isset($_SESSION['cart'][$product_id])) {
            unset($_SESSION['cart'][$product_id]);
            $success = "Product removed from cart";
        }
    }
    
    if(isset($_POST['update_quantity'])) {
        $product_id = (int)$_POST['product_id'];
        $quantity = max(1, (int)$_POST['quantity']);
        
        if(isset($_SESSION['cart'][$product_id])) {
            $_SESSION['cart'][$product_id]['quantity'] = $quantity;
        }
    }
    
    if(isset($_POST['clear_cart'])) {
        $_SESSION['cart'] = [];
        $success = "Cart cleared successfully";
    }
    
    if(isset($_POST['checkout'])) {
        // Get cart items from database
        if(!empty($_SESSION['cart'])) {
            $product_ids = array_keys($_SESSION['cart']);
            $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
            
            $stmt = $conn->prepare("SELECT * FROM products WHERE id IN ($placeholders) AND status = 'active'");
            $stmt->bind_param(str_repeat('i', count($product_ids)), ...$product_ids);
            $stmt->execute();
            $cart_products = $stmt->get_result();
            
            $total_amount = 0;
            $cart_items = [];
            
            while($product = $cart_products->fetch_assoc()) {
                $quantity = $_SESSION['cart'][$product['id']]['quantity'];
                $total_amount += $product['sale_price'] * $quantity;
                $cart_items[] = $product;
            }
            
            $_SESSION['checkout_data'] = [
                'items' => $cart_items,
                'total' => $total_amount
            ];
            
            header("Location: checkout.php");
            exit();
        }
    }
}

// Get cart products
$cart_items = [];
$total_amount = 0;

if(!empty($_SESSION['cart'])) {
    $product_ids = array_keys($_SESSION['cart']);
    if(!empty($product_ids)) {
        $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
        
        $stmt = $conn->prepare("SELECT * FROM products WHERE id IN ($placeholders) AND status = 'active'");
        $stmt->bind_param(str_repeat('i', count($product_ids)), ...$product_ids);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while($product = $result->fetch_assoc()) {
            $quantity = $_SESSION['cart'][$product['id']]['quantity'];
            $product['quantity'] = $quantity;
            $product['subtotal'] = $product['sale_price'] * $quantity;
            $cart_items[] = $product;
            $total_amount += $product['subtotal'];
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shopping Cart - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .cart-item {
            transition: all 0.3s ease;
        }
        .cart-item:hover {
            background-color: #f9fafb;
        }
        .quantity-btn {
            transition: all 0.2s ease;
        }
        .quantity-btn:hover {
            transform: scale(1.1);
        }
        .empty-cart {
            animation: bounce 2s infinite;
        }
        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="products.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Products</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">Shopping Cart</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-shopping-cart mr-2"></i> Shopping Cart
                </h1>
                <p class="text-gray-600">Review your items and proceed to checkout</p>
            </div>
            
            <!-- Messages -->
            <?php if(isset($success)): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Cart Content -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Cart Items -->
                <div class="lg:col-span-2">
                    <?php if(!empty($cart_items)): ?>
                        <!-- Cart Items List -->
                        <div class="bg-white rounded-xl shadow-lg overflow-hidden mb-6">
                            <div class="px-6 py-4 border-b bg-gradient-to-r from-blue-50 to-indigo-50">
                                <div class="flex items-center justify-between">
                                    <h3 class="text-xl font-bold text-gray-800">
                                        <i class="fas fa-boxes mr-2"></i> Cart Items (<?php echo count($cart_items); ?>)
                                    </h3>
                                    <form method="POST" action="" onsubmit="return confirm('Are you sure you want to clear your cart?');">
                                        <button type="submit" name="clear_cart" 
                                                class="text-red-600 hover:text-red-700 font-medium text-sm">
                                            <i class="fas fa-trash-alt mr-1"></i> Clear Cart
                                        </button>
                                    </form>
                                </div>
                            </div>
                            
                            <div class="divide-y">
                                <?php foreach($cart_items as $item): ?>
                                <div class="cart-item p-6">
                                    <div class="flex flex-col md:flex-row md:items-center">
                                        <!-- Product Image -->
                                        <div class="md:w-32 md:h-32 w-full h-48 mb-4 md:mb-0 md:mr-6 rounded-lg overflow-hidden">
                                            <img src="<?php echo $item['image_url'] ?: 'https://via.placeholder.com/300'; ?>" 
                                                 alt="<?php echo htmlspecialchars($item['name']); ?>"
                                                 class="w-full h-full object-cover hover:scale-110 transition duration-300">
                                        </div>
                                        
                                        <!-- Product Info -->
                                        <div class="flex-1">
                                            <div class="flex flex-col md:flex-row md:items-start justify-between">
                                                <div class="mb-4 md:mb-0">
                                                    <h4 class="font-bold text-gray-800 text-lg mb-2">
                                                        <a href="product-details.php?id=<?php echo $item['id']; ?>" 
                                                           class="hover:text-blue-600 transition duration-300">
                                                            <?php echo htmlspecialchars($item['name']); ?>
                                                        </a>
                                                    </h4>
                                                    <p class="text-gray-600 text-sm mb-2">
                                                        <?php echo substr(strip_tags($item['description']), 0, 100); ?>...
                                                    </p>
                                                    <div class="flex items-center">
                                                        <span class="text-gray-700 font-medium">
                                                            Unit Price: 
                                                            <span class="text-blue-600 font-bold">
                                                                ৳<?php echo number_format($item['sale_price'], 2); ?>
                                                            </span>
                                                        </span>
                                                    </div>
                                                </div>
                                                
                                                <!-- Price -->
                                                <div class="text-right">
                                                    <p class="text-2xl font-bold text-gray-800 mb-2">
                                                        ৳<?php echo number_format($item['subtotal'], 2); ?>
                                                    </p>
                                                    <?php if($item['original_price'] > 0): ?>
                                                        <p class="text-gray-400 line-through text-sm">
                                                            ৳<?php echo number_format($item['original_price'] * $item['quantity'], 2); ?>
                                                        </p>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            
                                            <!-- Actions -->
                                            <div class="flex items-center justify-between mt-4 pt-4 border-t">
                                                <!-- Quantity Controls -->
                                                <div class="flex items-center">
                                                    <form method="POST" action="" class="flex items-center">
                                                        <input type="hidden" name="product_id" value="<?php echo $item['id']; ?>">
                                                        <button type="button" 
                                                                onclick="updateQuantity(<?php echo $item['id']; ?>, -1)"
                                                                class="quantity-btn w-8 h-8 bg-gray-100 rounded-l-lg flex items-center justify-center hover:bg-gray-200">
                                                            <i class="fas fa-minus text-gray-600"></i>
                                                        </button>
                                                        
                                                        <input type="number" 
                                                               id="quantity_<?php echo $item['id']; ?>"
                                                               name="quantity" 
                                                               value="<?php echo $item['quantity']; ?>" 
                                                               min="1" 
                                                               max="99"
                                                               class="w-16 h-8 text-center border-y border-gray-300 bg-white"
                                                               onchange="updateQuantityInput(<?php echo $item['id']; ?>, this.value)">
                                                        
                                                        <button type="button" 
                                                                onclick="updateQuantity(<?php echo $item['id']; ?>, 1)"
                                                                class="quantity-btn w-8 h-8 bg-gray-100 rounded-r-lg flex items-center justify-center hover:bg-gray-200">
                                                            <i class="fas fa-plus text-gray-600"></i>
                                                        </button>
                                                        
                                                        <button type="submit" name="update_quantity" 
                                                                class="ml-3 text-blue-600 hover:text-blue-700 font-medium text-sm">
                                                            <i class="fas fa-sync-alt mr-1"></i> Update
                                                        </button>
                                                    </form>
                                                </div>
                                                
                                                <!-- Remove Button -->
                                                <form method="POST" action="" 
                                                      onsubmit="return confirm('Remove this item from cart?');">
                                                    <input type="hidden" name="product_id" value="<?php echo $item['id']; ?>">
                                                    <button type="submit" name="remove_item" 
                                                            class="text-red-600 hover:text-red-700 font-medium">
                                                        <i class="fas fa-trash-alt mr-1"></i> Remove
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <!-- Continue Shopping -->
                        <div class="flex justify-between">
                            <a href="products.php" 
                               class="inline-flex items-center text-blue-600 hover:text-blue-700 font-medium">
                                <i class="fas fa-arrow-left mr-2"></i> Continue Shopping
                            </a>
                            
                            <div class="text-right">
                                <p class="text-gray-600 text-sm">Items in cart: <?php echo count($cart_items); ?></p>
                            </div>
                        </div>
                        
                    <?php else: ?>
                        <!-- Empty Cart -->
                        <div class="bg-white rounded-xl shadow-lg p-12 text-center animate__animated animate__fadeIn">
                            <div class="empty-cart w-32 h-32 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                <i class="fas fa-shopping-cart text-gray-400 text-5xl"></i>
                            </div>
                            <h3 class="text-2xl font-bold text-gray-700 mb-4">Your cart is empty</h3>
                            <p class="text-gray-600 mb-8 max-w-md mx-auto">
                                Looks like you haven't added any products to your cart yet.
                            </p>
                            <div class="flex flex-col sm:flex-row justify-center gap-4">
                                <a href="products.php" 
                                   class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-8 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                    <i class="fas fa-shopping-cart mr-2"></i> Browse Products
                                </a>
                                <a href="dashboard.php" 
                                   class="bg-gray-200 text-gray-700 font-bold py-3 px-8 rounded-lg hover:bg-gray-300 transition duration-300">
                                    <i class="fas fa-tachometer-alt mr-2"></i> Go to Dashboard
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Order Summary -->
                <div class="space-y-6">
                    <?php if(!empty($cart_items)): ?>
                    <!-- Order Summary Card -->
                    <div class="bg-white rounded-xl shadow-lg p-6 sticky top-24">
                        <h3 class="text-xl font-bold text-gray-800 mb-6">
                            <i class="fas fa-receipt mr-2"></i> Order Summary
                        </h3>
                        
                        <!-- Summary Details -->
                        <div class="space-y-4 mb-6">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Subtotal</span>
                                <span class="font-medium text-gray-800">
                                    ৳<?php echo number_format($total_amount, 2); ?>
                                </span>
                            </div>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">Discount</span>
                                <span class="font-medium text-green-600">৳0.00</span>
                            </div>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">Platform Fee</span>
                                <span class="font-medium text-gray-800">৳0.00</span>
                            </div>
                            
                            <div class="border-t pt-4">
                                <div class="flex justify-between items-center">
                                    <span class="text-lg font-bold text-gray-800">Total Amount</span>
                                    <span class="text-3xl font-bold text-blue-600">
                                        ৳<?php echo number_format($total_amount, 2); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- User Balance -->
                        <div class="mb-6 p-4 bg-gray-50 rounded-lg">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-gray-600">Your Balance:</span>
                                <span class="text-xl font-bold <?php echo getUserBalance($user_id) >= $total_amount ? 'text-green-600' : 'text-red-600'; ?>">
                                    ৳<?php echo number_format(getUserBalance($user_id), 2); ?>
                                </span>
                            </div>
                            
                            <?php if(getUserBalance($user_id) < $total_amount): ?>
                                <div class="mt-3 p-3 bg-red-50 border border-red-200 rounded-lg">
                                    <div class="flex items-center">
                                        <i class="fas fa-exclamation-triangle text-red-500 mr-3"></i>
                                        <div>
                                            <p class="text-red-700 text-sm font-medium">Insufficient Balance</p>
                                            <p class="text-red-600 text-xs">
                                                Need ৳<?php echo number_format($total_amount - getUserBalance($user_id), 2); ?> more
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Checkout Button -->
                        <form method="POST" action="">
                            <button type="submit" name="checkout"
                                    class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-4 px-6 rounded-lg text-lg hover:from-green-600 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed"
                                    <?php echo getUserBalance($user_id) < $total_amount ? 'disabled' : ''; ?>>
                                <i class="fas fa-lock mr-2"></i> 
                                <?php echo getUserBalance($user_id) >= $total_amount ? 'Proceed to Checkout' : 'Insufficient Balance'; ?>
                            </button>
                        </form>
                        
                        <?php if(getUserBalance($user_id) < $total_amount): ?>
                            <div class="mt-4">
                                <a href="deposit.php" 
                                   class="block w-full text-center bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                    <i class="fas fa-plus-circle mr-2"></i> Deposit Now
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Payment Methods -->
                        <div class="mt-6 pt-6 border-t">
                            <h4 class="font-medium text-gray-700 mb-3">
                                <i class="fas fa-credit-card mr-2"></i> Secure Payment
                            </h4>
                            <div class="flex justify-center space-x-4">
                                <div class="text-center">
                                    <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-2">
                                        <i class="fas fa-wallet text-blue-600 text-xl"></i>
                                    </div>
                                    <p class="text-xs text-gray-600">Wallet</p>
                                </div>
                                <div class="text-center">
                                    <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-2">
                                        <i class="fas fa-shield-alt text-green-600 text-xl"></i>
                                    </div>
                                    <p class="text-xs text-gray-600">Secure</p>
                                </div>
                                <div class="text-center">
                                    <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-2">
                                        <i class="fas fa-bolt text-purple-600 text-xl"></i>
                                    </div>
                                    <p class="text-xs text-gray-600">Instant</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Support Card -->
                    <div class="bg-gradient-to-r from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-white">
                        <h3 class="text-xl font-bold mb-4">
                            <i class="fas fa-question-circle mr-2"></i> Need Help?
                        </h3>
                        <p class="mb-6 opacity-90">
                            Our support team is available 24/7 to assist you with any questions.
                        </p>
                        <div class="space-y-3">
                            <a href="https://t.me/support" target="_blank" 
                               class="flex items-center justify-center bg-white text-blue-600 font-bold py-3 px-4 rounded-lg hover:bg-blue-50 transition duration-300">
                                <i class="fab fa-telegram mr-2 text-xl"></i> Telegram Support
                            </a>
                            <a href="https://wa.me/8801234567890" target="_blank" 
                               class="flex items-center justify-center bg-white text-green-600 font-bold py-3 px-4 rounded-lg hover:bg-green-50 transition duration-300">
                                <i class="fab fa-whatsapp mr-2 text-xl"></i> WhatsApp Support
                            </a>
                        </div>
                    </div>
                    
                    <!-- Promo Code -->
                    <?php if(!empty($cart_items)): ?>
                    <div class="bg-white rounded-xl shadow-lg p-6">
                        <h3 class="text-xl font-bold text-gray-800 mb-4">
                            <i class="fas fa-tag mr-2"></i> Promo Code
                        </h3>
                        <div class="space-y-3">
                            <div class="flex">
                                <input type="text" 
                                       class="flex-1 px-4 py-3 border border-gray-300 rounded-l-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                       placeholder="Enter promo code">
                                <button class="bg-gradient-to-r from-purple-500 to-pink-600 text-white font-bold px-6 rounded-r-lg hover:opacity-90 transition duration-300">
                                    Apply
                                </button>
                            </div>
                            <p class="text-gray-500 text-xs">
                                Enter promo code to get discounts on your purchase.
                            </p>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Quantity Functions
    function updateQuantity(productId, change) {
        const quantityInput = document.getElementById('quantity_' + productId);
        let newQuantity = parseInt(quantityInput.value) + change;
        
        if(newQuantity < 1) newQuantity = 1;
        if(newQuantity > 99) newQuantity = 99;
        
        quantityInput.value = newQuantity;
    }
    
    function updateQuantityInput(productId, value) {
        let newValue = parseInt(value);
        
        if(isNaN(newValue) || newValue < 1) newValue = 1;
        if(newValue > 99) newValue = 99;
        
        document.getElementById('quantity_' + productId).value = newValue;
    }
    
    // Show notification
    function showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Add animation to empty cart icon
    document.addEventListener('DOMContentLoaded', function() {
        const emptyCartIcon = document.querySelector('.empty-cart');
        if(emptyCartIcon) {
            setInterval(() => {
                emptyCartIcon.classList.toggle('animate__bounce');
            }, 2000);
        }
    });
    </script>
</body>
</html>